﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.IO.Ports;
using System.Threading;

namespace VirtualSerialPortSample_dotNet
{
    public partial class MainForm : Form
    {
        private SerialPort comPort = null;

        private byte[] queryCommand = new byte[] { 0x1b, 0x06, 0x01 };

        private const int timeout = 5000;
        private const int STATUS_LENGTH = 9;

        public MainForm()
        {
            this.StartPosition = FormStartPosition.CenterScreen;

            InitializeComponent();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            textBoxFilePath.Text = Directory.GetCurrentDirectory() + Path.DirectorySeparatorChar.ToString() + "printData_StarLine.bin";
            textBoxFilePath.SelectionStart = textBoxFilePath.Text.Length;

            string[] portName = SerialPort.GetPortNames();

            if (portName != null)
            {
                comboBoxPorts.Items.AddRange(portName);
                comboBoxPorts.SelectedIndex = 0;
            }

            comboBoxFlowControl.Items.AddRange(new string[] { "DTR", "None" });
            comboBoxFlowControl.SelectedIndex = 0;
        }

        private void buttonSend_Click(object sender, EventArgs e)
        {
            try
            {
                this.Enabled = false;

                //EN: Set com port
                //JP: ポート設定
                comPort = new SerialPort(comboBoxPorts.SelectedItem.ToString(), 9600, Parity.None, 8, StopBits.One);
                comPort.Handshake = (comboBoxFlowControl.SelectedItem.ToString() == "DTR") ? Handshake.RequestToSend : Handshake.None;

                comPort.ReadTimeout = (int)(numericUpDownReadTimeout.Value) * 1000;
                comPort.WriteTimeout = (int)(numericUpDownWriteTimeout.Value) * 1000;

                //EN: Get printer status
                //JP: 印刷データ取得
                byte[] printData = File.ReadAllBytes(textBoxFilePath.Text);

                //EN: Add ETB command to the end of print data
                //JP: 印刷データの末尾にETBコマンドを付加する
                Array.Resize<byte>(ref printData, printData.Length + 1);
                printData[printData.Length - 1] = 0x17;

                //EN: Open port
                //JP: ポートオープン
                comPort.Open();
                Logging(comPort.PortName + " opened.");

                //EN: Send printing data
                //JP: データ送信実行
                while (true)
                {
                    bool result = false;

                    result = SendDataToPrinter(printData);

                    if (result == true)
                        break;

                    //Retry
                    if (MessageBox.Show(this, "Failed.", "Error", MessageBoxButtons.RetryCancel, MessageBoxIcon.Error) == DialogResult.Cancel)
                        break;
                }
            }
            catch (Exception ex)
            {
                Logging(ex.Message);
                MessageBox.Show(this, ex.Message, ex.Source);
            }
            finally
            {
                Logging(comPort.PortName + " closed.\r\n");
                comPort.Close();
                comPort = null;

                this.Enabled = true;
            }
        }

        private bool SendDataToPrinter(byte[] printData)
        {
            // EN: Check printer status
            // JP: ステータス要求
            Logging("Query Printer Status...");
            byte[] beforeStatus = new byte[STATUS_LENGTH];
            if (GetPrinterStatus(out beforeStatus) == false)
            {
                Logging("GetStarPrinterStatus() failed.");
                return false;
            }

            if ((beforeStatus[2] & 0x08) == 0x08)
            {
                Logging("Printer off-line detected.");
                Logging("Failed.");
                return false;
            }

            // EN: Send data
            // JP: データ送信
            Logging("Sending data");
            comPort.Write(printData, 0, printData.Length);

            // EN: Check printer status (Repeat)
            // JP: ステータス要求 (ポーリング)
            Logging("Query Printer Status...");
            byte[] afterStatus = new byte[STATUS_LENGTH];
            bool result = false;
            int time = 0;

            while (true)
            {
                if (GetPrinterStatus(out afterStatus) == false)
                {
                    Logging("GetStarPrinterStatus() failed.");
                    break;
                }

                //EN: Repetition interval: 500msec
                //JP: ポーリング間隔: 500msec
                Thread.Sleep(500);
                time += 500;

                if (beforeStatus[7] != afterStatus[7]) //EN: Count ETB counter up
                {                                      //JP: ETBカウントアップ
                    Logging("ETB count up.");
                    Logging("Success.");
                    result = true;
                    break;
                }

                if ((afterStatus[2] & 0x08) == 0x08) // Printer Offline
                {
                    Logging("Printer off-line detected.");
                    Logging("Failed.");
                    break;
                }

                if (timeout <= time) //Timeout
                {
                    Logging("Status polling timeout.");
                    Logging("Failed.");
                    break;
                }
            }

            return result;
        }

        private bool GetPrinterStatus(out byte[] status)
        {
            // EN: Change flow control disable in the meantime
            // JP: 一時的にフロー制御を無効にする
            Handshake handshake = comPort.Handshake;
            comPort.Handshake = Handshake.None;

            try
            {
                // EN: Send status request command
                // JP: ステータス要求コマンド送信
                comPort.Write(queryCommand, 0, queryCommand.Length);

                Thread.Sleep(200);

                // EN: Get status
                // JP: ステータス取得
                byte[] readBuffer = new byte[STATUS_LENGTH];
                int readSize = comPort.Read(readBuffer, 0, readBuffer.Length);

                if (readSize != STATUS_LENGTH)
                {
                    Logging("Read failed.");
                    status = null;
                    return false;
                }

                status = readBuffer;
                ShowPrinterStatus(status);

                return true;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, ex.Source);
                status = null;
                return false;
            }
            finally
            {
                //EN: Reset flow control to enable from disable
                //JP: フロー制御を元に戻す
                comPort.Handshake = handshake;
            }
        }

        private void Logging(string message)
        {
            textBoxLog.AppendText(message + "\r\n");

            textBoxLog.SelectionStart = textBoxLog.TextLength;
            textBoxLog.ScrollToCaret();
        }

        private void ShowPrinterStatus(byte[] status)
        {
            textBoxLog.AppendText("  ");
            for (int i = 0; i < status.Length; i++)
            {
                textBoxLog.AppendText(status[i].ToString("X2") + ", ");
            }
            textBoxLog.Text += "\r\n";

            textBoxLog.SelectionStart = textBoxLog.TextLength;
            textBoxLog.ScrollToCaret();
        }

        private void buttonClear_Click(object sender, EventArgs e)
        {
            textBoxLog.Clear();
        }

        private void buttonQuit_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void buttonSelectFile_Click(object sender, EventArgs e)
        {
            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Filter = "All files (*.*)|*.*";

            if (dialog.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            textBoxFilePath.Text = dialog.FileName;
            textBoxFilePath.SelectionStart = textBoxFilePath.Text.Length;
        }
    }
}